import mongoose from "mongoose";
import Joi from "joi";

// TransactionHistory Schema
const TransactionHistorySchema = new mongoose.Schema({
    taskId: {
        type: mongoose.Schema.Types.ObjectId,
        ref: 'Task', // Reference to the Task model
        required: true
    },
    userId: {
        type: mongoose.Schema.Types.ObjectId,
        ref: 'UserWallet',
        required: true
    },
    referrerId: {
        type: mongoose.Schema.Types.ObjectId,
        ref: 'UserWallet',
        default: null
    },
    rewardAmount: {
        type: Number,
        required: true
    },
    referrerRewardAmount: {
        type: Number,
        default: 0 // 40% of the reward, if applicable
    },
    isUserClaimed: {
        type: Boolean,
        default: false
    },
    isReferralCompleted: {
        type: Boolean,
        default: false
    },
    isReferrerClaimed: {
        type: Boolean,
        default: false
    },
    isCorrectAnswer: {
        type: Boolean,
        default: false
    },
    taskType: {
        type: String,
        enum: ['MCQ', 'Referral', 'OpenLink'], // Updated enum to include 'OpenLink' and remove 'DailyCheckIn'
        required: true
    },
    mcqResponse: {
        type: String, // User's response to MCQ task, applicable if taskType is 'MCQ'
        required: function () { return this.taskType === 'MCQ'; }
    },
    isLinkOpened: {
        type: Boolean, // Whether the user opened the link, applicable if taskType is 'OpenLink'
        default: false,
        required: function () { return this.taskType === 'OpenLink'; }
    },
    referralCount: {
        type: Number, // Number of referrals completed, applicable if taskType is 'Referral'
        default: 0,
        required: function () { return this.taskType === 'Referral'; }
    },
    status: {
        type: String,
        enum: ['ongoing', 'completed'],
        default: 'ongoing' // Initially, the transaction is ongoing
    }
}, { timestamps: true });

// Validation function
function validateTransactionHistory(transaction) {
    const schema = Joi.object({
        taskId: Joi.string().required(),
        userId: Joi.string().required(),
        referrerId: Joi.string().optional(),
        rewardAmount: Joi.number().required(),
        referrerRewardAmount: Joi.number().optional(),
        isUserClaimed: Joi.boolean().optional(),
        isReferrerClaimed: Joi.boolean().optional(),
        isCorrectAnswer: Joi.boolean().optional(),
        taskType: Joi.string().valid('MCQ', 'Referral', 'OpenLink').required(),
        mcqResponse: Joi.string().optional().when('taskType', { is: 'MCQ', then: Joi.required() }),
        isLinkOpened: Joi.boolean().optional().when('taskType', { is: 'OpenLink', then: Joi.required() }),
        referralCount: Joi.number().optional().when('taskType', { is: 'Referral', then: Joi.required() }),
        status: Joi.string().valid('ongoing', 'completed').optional(),
    });

    return schema.validate(transaction);
}

const TransactionHistory = mongoose.model("TransactionHistory", TransactionHistorySchema);

export { TransactionHistory, validateTransactionHistory };
