import mongoose from "mongoose";
import Joi from "joi";

// Define a subdocument schema for MCQ details
const MCQSchema = new mongoose.Schema({
    question: {
        type: String,
        required: true
    },
    options: [
        {
            type: String,
            required: true
        }
    ],
    correctAnswer: {
        type: String,
        required: true
    }
});

// Task Schema
const TaskSchema = new mongoose.Schema({
    description: {
        type: String,
    },
    reward_amount: {
        type: Number,
        default: 0
    },
    type: {
        type: String,
        enum: ['MCQ', 'OpenLink', 'Referral'],
        required: true
    },
    mcq: {
        type: MCQSchema, // Subdocument for MCQ tasks
        required: function () { return this.type === 'MCQ'; }
    },
    openLink: {
        url: {
            type: String,
            required: function () { return this.type === 'OpenLink'; }
        }
    },
    referralTarget: {
        type: Number, // Number of referrals required for Referral tasks
        default: 0,
        required: function () { return this.type === 'Referral'; }
    }
}, { timestamps: true });

// Validation function for Task
function validateTask(task) {
    const schema = Joi.object({
        //name: Joi.string().required(),
        description: Joi.string().optional(),
        reward_amount: Joi.number().required(),
        type: Joi.string().valid('MCQ', 'OpenLink', 'Referral').required(),
        // status: Joi.boolean().optional(),
        // rewardClaimed: Joi.boolean().optional(),
        // Ensuring mcq object is required only when type is MCQ
        mcq: Joi.object({
            question: Joi.string().required(),
            options: Joi.array().items(Joi.string().required()).min(2).required(),
            correctAnswer: Joi.string().required()
        }).optional().when('type', { is: 'MCQ', then: Joi.required() }),
        // Ensuring openLink is required only when type is OpenLink
        openLink: Joi.object({
            url: Joi.string().uri().required()
        }).optional().when('type', { is: 'OpenLink', then: Joi.required() }),
        // Ensuring referralTarget is required only when type is Referral
        referralTarget: Joi.number().allow('').optional().when('type', { is: 'Referral', then: Joi.required() }),
    });

    return schema.validate(task);
}

const Task = mongoose.model("Task", TaskSchema);

export { Task, validateTask };
